# Company:  Delta Elektronika B.V.
# Project:  INT-MOD-ANY_Examples
# File:     EthernetIP.py
# Date:     19/05/2025

# This code is provided "as is" without any guarantees or warranties. Delta Elektronika B.V.
# is not responsible for any damages, losses, or issues arising from its use, implementation, or modification.

from pycomm3 import CIPDriver, Services, BYTE
import struct

"""
Class: EthernetIP

Description:
	A nested class within INT-MOD-ANY that interacts with and controls
	a Delta Elektronika power supply using the Ethernet/IP (Class 3) protocol.
	Communication speed of this class is set-up automatically according to fieldbus speed set on the INT-MOD-ANY.
"""

class EthernetIP:
	def __init__(self, address, dataformat):
		self.address = address                                                                                          # Save entered address in variable.
		self.client = None                                                                                              # define the Ethernet/IP client.
		self.dataformat = dataformat
		self.cvprg_buffer = None                                                                                        # These values must be combined before sending out, since the Anybus module expects a bytestring.
		self.ccprg_buffer = None                                                                                        # ..
		self.remctrl_buffer = None                                                                                      # ..
		if self.dataformat == 'float_format_a':
			self.cvprg_buffer = b'\x00\x00\x00\x00'                                                                     # Init local variables
			self.ccprg_buffer = b'\x00\x00\x00\x00'                                                                     # ..
			self.remctrl_buffer = b'\x00\x00'                                                                           # ..
		elif self.dataformat == '16bit_format_a':                                                                       # ..
			self.cvprg_buffer = b'\x00\x00'                                                                             # ..
			self.ccprg_buffer = b'\x00\x00'                                                                             # ..
			self.remctrl_buffer = b'\x00\x00'                                                                           # ..


	def connect(self):                                                                                                  # Connect to the Ethernet/IP bus.
		self.client = CIPDriver(self.address)                                                                           # ..
		self.client.open()                                                                                              # ..


	def request(self, parameter):                                                                                       # Request a parameter value from the Delta Elektronika powersupply using Ethernet/IP:

		if self.dataformat == 'float_format_a':                                                                         # Using Float Format.
			requested_data = self.get_client_data(0x04, 0x64, 0x03, BYTE[30])   # ..
			for i in range(30 * 8):                                                                                     # ..
				requested_data[i] = 1 if (requested_data[i]) else 0                                                     # ..
			match parameter.lower():                                                                                    # ..
				case 'cvprg':                                                                                           # ..
					return self.array_to_float(requested_data,0,32)                                       # ..
				case 'cvmon':                                                                                           # ..
					return self.array_to_float(requested_data, 32, 32)                                   	# ..
				case 'ccprg':                                                                                           # ..
					return self.array_to_float(requested_data,64, 32)                                     # ..
				case 'ccmon':                                                                                           # ..
					return self.array_to_float(requested_data, 96, 32)                                  	# ..
				case 'cvlim':                                                                                           # ..
					return self.array_to_float(requested_data, 128, 32)                                 	# ..
				case 'cclim':                                                                                           # ..
					return self.array_to_float(requested_data, 160, 32)                                 	# ..
				case 'refresh':                                                                                         # ..
					return self.array_to_int(requested_data, 192, 16)                                   	# ..
				case 'status_registera':                                                                                # ..
					return self.array_to_int(requested_data, 208, 16)                                   	# ..
				case 'status_registerb':                                                                                # ..
					return self.array_to_int(requested_data, 224, 16)                                   	# ..
				case _:
					raise ValueError(f"Incorrect parameter has been requested: {parameter}")                            # Error

		elif self.dataformat == '16bit_format_a':
			requested_data = self.get_client_data(0x04, 0x64, 0x03, BYTE[12])   # Using 16-Bit Format.
			for i in range(12 * 8):                                                                                     # ..
				requested_data[i] = 1 if (requested_data[i]) else 0                                                     # ..
			match parameter.lower():                                                                                    # ..
				case 'cvprg':                                                                                           # ..
					return self.array_to_int(requested_data,0,16)                                        	# ..
				case 'ccprg':                                                                                           # ..
					return self.array_to_int(requested_data,32, 16)                                      	# ..
				case 'cvmon':                                                                                           # ..
					return self.array_to_int(requested_data, 16, 16)                                     	# ..
				case 'ccmon':                                                                                           # ..
					return self.array_to_int(requested_data, 48, 16)                                     	# ..
				case 'refresh':                                                                                         # ..
					return self.array_to_int(requested_data, 64, 16)                                     	# ..
				case 'status_registera':                                                                                # ..
					return self.array_to_int(requested_data, 80, 16)                                     	# ..
				case _:
					raise ValueError(f"Incorrect parameter has been requested: {parameter}")                            # Error


	def set(self, parameter, value):                                                                                    # Set a parameter value of the Delta Elektronika powersupply using Ethernet/IP:
		if self.dataformat == 'float_format_a':                                                                         # Using Float Format.
			match parameter.lower():                                                                                    # ..
				case 'cvprg':                                                                                           # Set CVprg:
					self.cvprg_buffer = self.float_to_struct(value)                                                     # convert 'value' argument to a bytestring and save in cvprg_buffer.
					combined = self.cvprg_buffer + self.ccprg_buffer + self.remctrl_buffer                              # Combine CVprg, CCprg, POWprg and STS into 1 bytestring.
					self.set_client_data(combined, 0x04, 0x96, 0x03, BYTE[10])  # ..
				case 'ccprg':                                                                                           # Set CCprg:
					self.ccprg_buffer = self.float_to_struct(value)                                                     # convert 'value' argument to a bytestring and save in cvprg_buffer.
					combined = self.cvprg_buffer + self.ccprg_buffer + self.remctrl_buffer                              # Combine CVprg, CCprg, POWprg and STS into 1 bytestring.
					self.set_client_data(combined, 0x04, 0x96, 0x03, BYTE[10])  # ..
				case 'remctrl':                                                                                         # Set RemCtrl:
					self.remctrl_buffer = self.int_to_struct(value)                                                     # convert 'value' argument to a bytestring and save in cvprg_buffer.
					combined = self.cvprg_buffer + self.ccprg_buffer + self.remctrl_buffer                              # Combine CVprg, CCprg, POWprg and STS into 1 bytestring.
					self.set_client_data(combined, 0x04, 0x96, 0x03, BYTE[10])  # ..
				case _:
					raise ValueError(f"Incorrect parameter is being set: {parameter}")                                  # Error

		elif self.dataformat == '16bit_format_a':                                                                       # Using 16-Bit Format:
			match parameter.lower():                                                                                    # ..
				case 'cvprg':                                                                                           # Set CVprg:
					self.cvprg_buffer = self.int_to_struct(value)                                                       # convert 'value' argument to a bytestring and save in cvprg_buffer.
					combined = self.cvprg_buffer + self.ccprg_buffer + self.remctrl_buffer                              # Combine CVprg, CCprg and STS into 1 bytestring.
					self.set_client_data(combined, 0x04, 0x96, 0x03, BYTE[6])   # ..
				case 'ccprg':                                                                                           # Set CCprg:
					self.ccprg_buffer = self.int_to_struct(value)                                                       # convert 'value' argument to a bytestring and save in cvprg_buffer.
					combined = self.cvprg_buffer + self.ccprg_buffer + self.remctrl_buffer                              # Combine CVprg, CCprg and STS into 1 bytestring.
					self.set_client_data(combined, 0x04, 0x96, 0x03, BYTE[6])   # ..
				case 'remctrl':                                                                                         # Set RemCtrl:
					self.remctrl_buffer = self.int_to_struct(value)                                                     # convert 'value' argument to a bytestring and save in cvprg_buffer.
					combined = self.cvprg_buffer + self.ccprg_buffer + self.remctrl_buffer                              # Combine CVprg, CCprg and STS into 1 bytestring.
					self.set_client_data(combined, 0x04, 0x96, 0x03, BYTE[6])   # ..


	def disconnect(self):                                                                                               # Disconnect from the Ethernet/IP bus.
		self.client.close()                                                                                             # ..


	def get_client_data(self, anybus_object, anybus_instance, anybus_attribute, anybus_datatype):           # Acquire the data on the passed object->instance->attribute.
		response = self.client.generic_message(                                                             # ..
			service=Services.get_attribute_single,                                                          # ..
			class_code=anybus_object,                                                                       # .. >Object
			instance=anybus_instance,                                                                       # .. >Instance
			attribute=anybus_attribute,                                                                     # .. >Attribute
			data_type=anybus_datatype,                                                                      # .. >Datatype
			connected=True,                                                                                 # ..
			name = 'GetData'                                                                                # ..
		)                                                                                                   # ..
		if response:                                                                                        # ..
			return response.value                                                                           # ..
		else:                                                                                               # ..
			raise ValueError(f"Could not collect data - {response.error}")                                  # ..


	def set_client_data(self, value, anybus_object, anybus_instance, anybus_attribute, anybus_datatype):    # Set the data on the passed object->instance->attribute.
		response = self.client.generic_message(                                                             # ..
			service=Services.set_attribute_single,                                                          # ..
			request_data=value,                                                                             # ..
			class_code=anybus_object,                                                                       # .. >Object
			instance=anybus_instance,                                                                       # .. >Instance
			attribute=anybus_attribute,                                                                     # .. >Attribute
			data_type=anybus_datatype,                                                                      # .. >Datatype
			connected=True,                                                                                 # ..
			name='SetData'                                                                                  # ..
		)
		return response.error


	def float_to_struct(self, value):                                                                       # Pack float to struct.
		packed = struct.pack('<f', value)                                                           # ..
		return packed                                                                                       # ..


	def int_to_struct(self, value):                                                                         # Pack integer to struct.
		packed = value.to_bytes(2, byteorder='little')                                                      # ..
		return packed                                                                                       # ..


	def array_to_float(self, buffer, start_bit, length):                                                    # Convert part of an array into a float.
		fourbytes = buffer[start_bit:start_bit+length]                                                      # Extract the first 32 bits
		fourbytes = list(reversed(fourbytes))                                                               # Reverse the bits in 'fourbytes'.
		binary_string = ''.join(str(bit) for bit in fourbytes)                                              # Convert the list of bits to a binary string
		int_value = int(binary_string, 2)                                                                   # Convert the binary string to an integer
		float_value = struct.unpack('f', int_value.to_bytes(4, byteorder='little'))[0]         # Use struct to interpret the integer as a 32-bit float
		return float_value


	def array_to_int(self, buffer, start_bit, length):                                                      # Convert part of an array into an integer.
		bits = buffer[start_bit:start_bit+length]                                                           # Extract the bits needed from 'buffer'.
		bits = list(reversed(bits))                                                                         # Reverse the bits in 'buffer'.
		binary_string = ''.join(str(bit) for bit in bits)                                                   # Convert the list of bits to a binary string
		int_value = int(binary_string, 2)                                                                   # Convert the binary string to an integer
		return int_value



